% This code implements Case Study III in the manuscript titled "Assessing
% and Resolving Model Misspecifications in Metabolic Flux Analysis".
%
% Requires: model.xml from Sokolenko et al., BMC Systems Biology, 10:91, 
% 2016 (source https://github.com/ssokolen/omfapy/tree/master/analysis).
% Please place model.xml file in the same directory as this code.
%
% Written by: Rudiyanto Gunawan
% Date: 14 February 2017

clear
close all

% Adding the directory containing F-test.m
addpath ..

nrOfRuns = 100; % number of independent runs of the iterative procedure 
nrOfTests = 100; % number of repeated iterative procedure for a given stoichiometric model
nrOfExtraReactions = 8; % number of extra reactions
nrOfReactionsRemoved = 8; % numbers of omitted reactions

model = sbmlimport('model.xml');
[S,species,reactions] = getstoichmatrix(model);
S = full(S);

indexvE = [];
for k = 1:36 %The first 36 species were extracellular 
    indexvE = [indexvE find(S(k,:)~=0)]; % Find the indeces of exchange reactions
end

species = species(37:end); %Track only intracellular species
S = S(37:end,[indexvE setdiff(1:size(S,2),indexvE)]); % Restructure S into [S_E S_I]

indexE = find(sum(abs(S(:,1:length(indexvE))),2)~=0); % Find the species indices with exchange reactions
S = S([indexE' setdiff(1:size(S,1),indexE)],:); % Reorganize S so that the species with exchange reactions are at the top

% Rearranging species and reaction indeces
species = species([indexE' setdiff(1:size(S,1),indexE)]);
reactions = reactions([indexvE setdiff(1:size(S,2),indexvE)]);

% Drop species and reactions following Sokolenko et al. 
indexSp = [1:8 10:22 24:42 52:60]; % Species kept
species = species(indexSp);

indexRx = [1:8 10:23 25:83]; % Reactions kept
reactions = reactions(indexRx);

S = S(indexSp, indexRx);

% Partition S matrix
S_E = S(:,1:34);
S_I = S(:,35:end);
S_I2 = S_I(34:end,:);

% Flux data from Sokolenko et al. 
vE = [-1.066944173
    -34.92313775
    -17.7866446
    2.37612474
    2.24459406
    2.041656253
    -2.142324594
    -1.61192265
    0.333098841
    -0.312515867
    -7.751086422
    161.1574586
    7.076289524
    -0.335451099
    -2.389447025
    18.27037083
    0.973588956
    1.410947924
    -291.2119101
    2.606265159
    1.690021196
    0.86471197
    -1.371998905
    -3.997049092
    1.146063304
    1.929724953
    -32.85391871
    -2.832654816
    -0.930798119
    2.478875947
    1.588374837
    0.32464466
    1.049494219
    2.175155428];

% Standard deviation of the flux data
svE = [0.056377069
    1.338419435
    4.076024162
    0.391548655
    0.195599932
    0.130942264
    0.268613188
    0.109142287
    0.063218758
    0.037509474
    0.186835638
    3.931906058
    1.01946138
    0.412144938
    0.220144178
    1.957832291
    0.151991886
    0.109297657
    10.92802378
    0.25286843
    0.119562459
    0.053118649
    0.165148424
    0.165838307
    0.083703239
    0.152716578
    4.635581343
    0.170817535
    0.110910374
    0.365269525
    0.170353621
    0.066300785
    0.073124188
    0.132416673];

% Construct the variance-covariance matrix
Omega = diag(svE.^2);
Omega = S_E*Omega*S_E';
for k = 34:size(Omega,2)
    Omega(k,k) = 1e-2;
end
Omega = 0.5*(Omega+Omega');

% Compute the square root of the variance covariance matrix
[L,D] = eig(Omega);
P = L*sqrt(D)*inv(L);

%Perform GLS to obtain vI,GLS
X = P\S_I;
y = -P\S_E*vE;
vIref = ((X'*X)\X')*y;

medall = [];

for a = 1:nrOfRuns
    
    % Generate the true stoichiometric matrix by removing n_extra reactions
    index = randperm(size(S_I,2),size(S_I,2)-nrOfExtraReactions);
    Strue = S_I(:,index);
    Sextra = S_I(:,setdiff(1:size(S_I,2),index));
    vItrue = vIref(index);
    vIextra = vIref(setdiff(1:size(S_I,2),index));
    
    summary = [];
    for k = 1:nrOfTests
        % Generate data of SEvE
        SvE = -Strue*vItrue + P*randn(size(Strue,1),1);
        
        % Generate a reduced stoichometric matrix
        index = randperm(size(Strue,2),size(Strue,2)-nrOfReactionsRemoved);
        Sred = Strue(:,index);
        Somit = Strue(:,setdiff(1:size(Strue,2),index));
        vIred = vItrue(index);
        vIomit = vItrue(setdiff(1:size(Strue,2),index));
        
        % Recast variables in GLS regression framework
        y = -P\SvE;
        X = P\Sred;
        Z = P\[Sextra Somit];
        
        inRxn = [];
        outRxn = 1:size(Z,2);
        
        %% Perform iterative procedure using k = 1
        pall = [];
        for j = outRxn
            p1 = Ftest(y,[X Z(:,inRxn)],Z(:,j));
            pall = [pall p1];
        end
        index = find(pall<0.05);
        inRxn = [inRxn outRxn(index)];
        outRxn = setdiff(1:size(Z,2),inRxn);
        
        numRxn = 0;
        while length(inRxn)~=numRxn 
            numRxn = length(inRxn);
            pall = [];
            for j = outRxn
                p1 = Ftest(y,[X Z(:,inRxn)],Z(:,j));
                pall = [pall p1];
            end
            index = find(pall<0.05);
            inRxn = [inRxn outRxn(index)];
            outRxn = setdiff(1:size(Z,2),inRxn);
        end
        
        %% Perform iterative procedure using k = 1 then k = 2  
%         pall = [];
%         for j = outRxn
%             p1 = Ftest(y,[X Z(:,inRxn)],Z(:,j));
%             pall = [pall p1];
%         end
%         index = find(pall<0.05);
%         inRxn = [inRxn outRxn(index)];
%         outRxn = setdiff(1:size(Z,2),inRxn);
%         
%         numRxn = 0;
%         while length(inRxn)~=numRxn 
%             numRxn = length(inRxn);
%             pall = [];
%             for j = outRxn
%                 p1 = Ftest(y,[X Z(:,inRxn)],Z(:,j));
%                 pall = [pall p1];
%             end
%             index = find(pall<0.05);
%             inRxn = [inRxn outRxn(index)];
%             outRxn = setdiff(1:size(Z,2),inRxn);
%         end
%         
%         if length(outRxn) >= 2
%             
%             % Generate all possible pairwise combination of the remaining reactions
%             combRxn = combnk(outRxn,2);
%             pall = [];
%             for j = 1:size(combRxn,1)
%                 p2 = Ftest(y,[X Z(:,inRxn)],Z(:,combRxn(j,:)));
%                 pall = [pall p2];
%             end
%             index = find(pall<0.05);
%             index = unique(combRxn(index,:));
%             inRxn = [inRxn reshape(index,1,length(index))];
%             outRxn = setdiff(1:size(Z,2),inRxn);
%             
%             numRxn = 0;
%             while and(length(inRxn)~=numRxn,length(outRxn)>=2) 
%                 numRxn = length(inRxn);
%                 combRxn = combnk(outRxn,2);
%                 pall = [];
%                 for j = 1:size(combRxn,1)
%                     p2 = Ftest(y,[X Z(:,inRxn)],Z(:,combRxn(j,:)));
%                     pall = [pall p2];
%                 end
%                 index = find(pall<0.05);
%                 index = unique(combRxn(index,:));
%                 inRxn = [inRxn reshape(index,1,length(index))];
%                 outRxn = setdiff(1:size(Z,2),inRxn);
%             end
%         end
          
        Xtra = find(outRxn<=nrOfExtraReactions); 
        Remain = find(outRxn>nrOfExtraReactions); 
        nXtra = length(Xtra); % number of remaining extra reactions 
        nRem = length(Remain); % number of remaining missing reactions
        
        summary = [summary; nXtra nRem];
        
    end
    
    medXtra = median(summary(:,1));
    medRem = median(summary(:,2));
    
    medall = [medall; medXtra medRem];
end

results = mean(medall)